
#if NUNIT
#region Directives
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.UnitTests;
using NUnit.Framework;
//-----------------------***
using System;
using System.Collections.Generic;
using System.Transactions;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
    [TestFixture]
    [Microsoft.VisualStudio.TestTools.UnitTesting.TestClass]
    public class BOL_PendingSpecimenTest : BaseTest
    {
        [SetUp]
        protected void SetUp()
        {
            this.BaseSetUp();
        }

        #region Get Pending Tests

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void DoTestsExist_False_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = false;
                DeleteTableData();

                // Act
                var actual = PendingSpecimenTest.DoTestsExist(this.TestDivisionCode, PendingTestStatus.Pending);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void DoTestsExist_PendingStatus_True_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);

                // Act
                var actual = PendingSpecimenTest.DoTestsExist(this.TestDivisionCode, PendingTestStatus.Pending);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void DoTestsExist_PendingStatus_False_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = false;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Accepted, OrderableTest.ABORh);

                // Act
                var actual = PendingSpecimenTest.DoTestsExist(this.TestDivisionCode, PendingTestStatus.Pending);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void GetPendingSpecimenTests_PendingStatus_NoData_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = false;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Accepted, OrderableTest.ABORh);

                // Act
                var data = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var actual = data.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void GetPendingSpecimenTests_PendingStatus_Data_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABS);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.TAS);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.AGe);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.DAT);
                pstArranger.ArrangeXM(PendingTestStatus.Pending, OrderableTest.XM, PendingTestHelper.INCOMPATIBLE);
                // Act
                var data = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var actual = data.Count == 6;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }
        #endregion

        #region Reject Pending Tests


        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void RejectTest_Fail()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = false;

                // Act
                var actual = PendingSpecimenTest.RejectTests(null);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected without specifying a rejection comment
        public void RejectTest_No_Tests_Fail()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                PendingSpecimenTest.RejectTests(pendingTests);
                var rejectedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Rejected, null);
                var actual = rejectedTests.Count == 1;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected without specifying a rejection comment
        public void RejectXMTest_Fail()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.ArrangeXM(PendingTestStatus.Pending, OrderableTest.RepeatXM, PendingTestHelper.INCOMPATIBLE);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                PendingSpecimenTest.RejectTests(pendingTests);
                var rejectedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Rejected, null);
                var actual = rejectedTests.Count == 1;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void RejectTest_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.ReviewComment = "reject comment - test";
                }

                PendingSpecimenTest.RejectTests(pendingTests);

                var rejectedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Rejected, null);
                var actual = rejectedTests.Count == 1;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }
        #endregion

        #region Accept Pending Tests

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_AboRh_No_Tests_Fail()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = false;

                // Act                
                bool isWorkLoadDefined = false;
                var actual = PendingSpecimenTest.AcceptTests(this.TestDivisionCode, null, ref isWorkLoadDefined);

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected becase TestValidationStatus == Error
        public void AcceptTest_AboRh_ErrorValidation_Fail()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                foreach (var test in pendingTests)
                {
                    test.TestValidation.TestValidationStatus = TestValidationStatus.Error;
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Rejected, null);
                var actual = acceptedTests.Count == 1;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected becase OrderedTest == null
        public void AcceptTest_AboRh_Null_OrderedTest_Fail()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                foreach (var test in pendingTests)
                {
                    test.OrderedTest = null;
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Rejected, null);
                var actual = acceptedTests.Count == 1;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        [ExpectedException] // exception expected without an exception override comment
        public void AcceptTest_AboRh_NoExceptionOverride_Fail()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                    pendingTest.TestValidation.ExceptionForDisplayList.Clear();
                    var exceptionList = pendingTest.TestValidation.ExceptionForDisplayList;
                    exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                    "Expired Order Selected",
                                                                    Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                    "PT"));
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_AboRh_WithExceptions_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                    pendingTest.TestValidation.ExceptionForDisplayList.Clear();
                    var exceptionList = pendingTest.TestValidation.ExceptionForDisplayList;
                    exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                    "Expired Order Selected",
                                                                    Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                    "PT"));
                    exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.ABORh_discrepancy,
                                                                    "Discrepant Repeat ABO/Rh typing",
                                                                    "Original and repeat ABO/Rh interpretation do not match. You must resolve the discrepancy before this specimen may be used. Continue to save?",
                                                                    "PT"));
                    exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                    "ABO/Rh Mismatch",
                                                                    StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                    "PT"));

                    foreach (var exception in exceptionList)
                    {
                        exception.OverrideMessage = "Unit test override";
                    }
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.Completed));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_AboRh_NoWorkLoad_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = false;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(false);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABORh);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.TestValidation.TestValidationStatus = TestValidationStatus.Valid;
                    pendingTest.TestValidation.ExceptionForDisplayList.Clear();
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);
                var actual = isWorkLoadDefined;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.Completed));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_ABS_WithExceptions_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.ABS);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                    pendingTest.TestValidation.ExceptionForDisplayList.Clear();
                    var exceptionList = pendingTest.TestValidation.ExceptionForDisplayList;
                    exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                    "Expired Order Selected",
                                                                    Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                    "PT"));
                    exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                    "ABO/Rh Mismatch",
                                                                    StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                    "PT"));

                    foreach (var exception in exceptionList)
                    {
                        exception.OverrideMessage = "Unit test override";
                    }
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.Completed));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_TAS_WithExceptions_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.TAS);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var tasTest = pendingTests[0] as PendingSpecimenTasModel;
                var aboModel = tasTest.PendingSpecimenAboRhModel;
                var absModel = tasTest.PendingSpecimenAbsModel;

                // Add exceptins to AboRhModel
                aboModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                aboModel.TestValidation.ExceptionForDisplayList.Clear();
                var exceptionList = aboModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                "ABO/Rh Mismatch",
                                                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                "PT"));
                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                // Add exceptins to AbsModel
                absModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                absModel.TestValidation.ExceptionForDisplayList.Clear();
                exceptionList = absModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                "ABO/Rh Mismatch",
                                                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                "PT"));
                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                // Accept tests
                bool isWorkLoadDefined = false;
                pendingTests = new List<IPendingSpecimenTestModel>();
                pendingTests.Add(aboModel);
                pendingTests.Add(absModel);
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.Completed));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_TAS_WithExceptions_SaveAbs_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.TAS);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var tasTest = pendingTests[0] as PendingSpecimenTasModel;
                var aboModel = tasTest.PendingSpecimenAboRhModel;
                var absModel = tasTest.PendingSpecimenAbsModel;

                // Add exceptins to AboRhModel
                aboModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                aboModel.TestValidation.ExceptionForDisplayList.Clear();
                var exceptionList = aboModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                "ABO/Rh Mismatch",
                                                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                "PT"));
                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                // Add exceptins to AbsModel
                absModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                absModel.TestValidation.ExceptionForDisplayList.Clear();
                exceptionList = absModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                "ABO/Rh Mismatch",
                                                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                "PT"));
                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                // Accept tests
                bool isWorkLoadDefined = false;
                pendingTests = new List<IPendingSpecimenTestModel>();
                //pendingTests.Add(aboModel); // don't add abo/rh model
                pendingTests.Add(absModel);
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.PendingReview));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_TAS_WithExceptions_SaveAboRh_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.TAS);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var tasTest = pendingTests[0] as PendingSpecimenTasModel;
                var aboModel = tasTest.PendingSpecimenAboRhModel;
                var absModel = tasTest.PendingSpecimenAbsModel;

                // Add exceptins to AboRhModel
                aboModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                aboModel.TestValidation.ExceptionForDisplayList.Clear();
                var exceptionList = aboModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                "ABO/Rh Mismatch",
                                                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                "PT"));
                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                // Add exceptins to AbsModel
                absModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                absModel.TestValidation.ExceptionForDisplayList.Clear();
                exceptionList = absModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                "ABO/Rh Mismatch",
                                                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                "PT"));
                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                // Accept tests
                bool isWorkLoadDefined = false;
                pendingTests = new List<IPendingSpecimenTestModel>();
                pendingTests.Add(aboModel);
                //pendingTests.Add(absModel); // don't add/accept abs model
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.PendingReview));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_TAS_WithExceptions_SaveAboRh_RejectAbs_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.TAS);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var tasTest = pendingTests[0] as PendingSpecimenTasModel;
                var aboModel = tasTest.PendingSpecimenAboRhModel;
                var absModel = tasTest.PendingSpecimenAbsModel;
                absModel.ReviewComment = "Reject test";

                // Add exceptins to AboRhModel
                aboModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                aboModel.TestValidation.ExceptionForDisplayList.Clear();
                var exceptionList = aboModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                "ABO/Rh Mismatch",
                                                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                "PT"));
                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                // Accept tests
                bool isWorkLoadDefined = false;
                var acceptTests = new List<IPendingSpecimenTestModel>() { aboModel };
                var rejectTests = new List<IPendingSpecimenTestModel>() { absModel };
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, acceptTests, ref isWorkLoadDefined);
                PendingSpecimenTest.RejectTests(rejectTests);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.PartiallyCompleted));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_TAS_WithExceptions_SaveAbs_RejectAboRh_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.TAS);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var tasTest = pendingTests[0] as PendingSpecimenTasModel;
                var aboModel = tasTest.PendingSpecimenAboRhModel;
                var absModel = tasTest.PendingSpecimenAbsModel;
                aboModel.ReviewComment = "Reject test";

                // Add exceptins to AboRhModel
                absModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                absModel.TestValidation.ExceptionForDisplayList.Clear();
                var exceptionList = absModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                "ABO/Rh Mismatch",
                                                                StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                "PT"));
                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                // Accept tests
                bool isWorkLoadDefined = false;
                var acceptTests = new List<IPendingSpecimenTestModel>() { absModel };
                var rejectTests = new List<IPendingSpecimenTestModel>() { aboModel };
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, acceptTests, ref isWorkLoadDefined);
                PendingSpecimenTest.RejectTests(rejectTests);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.PartiallyCompleted));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_DAT_WithExceptions_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.Arrange(PendingTestStatus.Pending, OrderableTest.DAT);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);

                foreach (var pendingTest in pendingTests)
                {
                    pendingTest.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                    pendingTest.TestValidation.ExceptionForDisplayList.Clear();
                    var exceptionList = pendingTest.TestValidation.ExceptionForDisplayList;
                    exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                    "Expired Order Selected",
                                                                    Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                    "PT"));
                    exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Unit_ER_issued_testing_problem,
                                                                    "ABO/Rh Mismatch",
                                                                    StrRes.SysErrMsg.UC056.EmergencyUnitsIssued().ResString,
                                                                    "PT"));

                    foreach (var exception in exceptionList)
                    {
                        exception.OverrideMessage = "Unit test override";
                    }
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.Completed));
            }
        }

        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_XM_WithExceptions_Incompatible_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.ArrangeXM(PendingTestStatus.Pending, OrderableTest.RepeatXM, PendingTestHelper.INCOMPATIBLE);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var xmModel = pendingTests[0] as PendingSpecimenXMatchModel;
                var origBloodUnit = new BOL.BloodUnit(xmModel.BloodUnitModel.BloodUnitGuid);

                xmModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                xmModel.TestValidation.ExceptionForDisplayList.Clear();
                var exceptionList = xmModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));

                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.Completed));
                // Verify BloodUnitStatus 
                var updatedBloodUnit = new BOL.BloodUnit(xmModel.BloodUnitModel.BloodUnitGuid);
                Assert.IsTrue(origBloodUnit.BloodUnitStatus.UnitStatusCode == UnitStatusCode.CrossmatchedAssigned);
                Assert.IsTrue(updatedBloodUnit.BloodUnitStatus.UnitStatusCode == UnitStatusCode.Available ||
                              updatedBloodUnit.BloodUnitStatus.UnitStatusCode == UnitStatusCode.Limited);
            }
        }


        [Test]
        [Category("UC_115_AutomatedTesting")]
        public void AcceptTest_XM_WithExceptions_Compatible_Pass()
        {
            using (var scope = new TransactionScope())
            {
                // Arrange
                var expected = true;
                DeleteTableData();
                var pstArranger = new PendingSpecimenTestArranger(true);
                pstArranger.ArrangeXM(PendingTestStatus.Pending, OrderableTest.RepeatXM, PendingTestHelper.COMPATIBLE_SAFE_TO_TRANSFUSE_ID);

                // Act
                var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Pending, null);
                var xmModel = pendingTests[0] as PendingSpecimenXMatchModel;
                var origBloodUnit = new BOL.BloodUnit(xmModel.BloodUnitModel.BloodUnitGuid);

                xmModel.TestValidation.TestValidationStatus = TestValidationStatus.Warning;
                xmModel.TestValidation.ExceptionForDisplayList.Clear();
                var exceptionList = xmModel.TestValidation.ExceptionForDisplayList;
                exceptionList.Add(new ExceptionForDisplayModel(ExceptionType.Expired_task_processed,
                                                                "Expired Order Selected",
                                                                Common.StrRes.ConfMsg.UC056.ExpiredTaskProcessed().ResString,
                                                                "PT"));

                foreach (var exception in exceptionList)
                {
                    exception.OverrideMessage = "Unit test override";
                }

                bool isWorkLoadDefined = false;
                PendingSpecimenTest.AcceptTests(pstArranger.DivisionCode, pendingTests, ref isWorkLoadDefined);

                var acceptedTests = PendingSpecimenTest.GetPendingSpecimenTests(pstArranger.Patient, pstArranger.PatientSpecimenGuid, pstArranger.DivisionCode, PendingTestStatus.Accepted, null);
                var actual = acceptedTests.Count > 0;

                // Assert
                Assert.IsTrue(expected == actual);
                // Verify Task Status
                Assert.IsTrue(VerifyOrderedTestTaskStatus(pendingTests, TaskStatus.Completed));
                // Verify BloodUnitStatus 
                var updatedBloodUnit = new BOL.BloodUnit(xmModel.BloodUnitModel.BloodUnitGuid);
                Assert.IsTrue(origBloodUnit.BloodUnitStatus.UnitStatusCode == UnitStatusCode.CrossmatchedAssigned);
                Assert.IsTrue(updatedBloodUnit.BloodUnitStatus.UnitStatusCode == UnitStatusCode.CrossmatchedAssigned);
            }
        }

        private bool VerifyOrderedTestTaskStatus(IList<IPendingSpecimenTestModel> pendingSpecimenList, TaskStatus taskStatus)
        {
            foreach (var pendingTest in pendingSpecimenList)
            {
                if (pendingTest.OrderedTest != null)
                {
                    var orderedTest = new OrderedTest(pendingTest.OrderedTest.OrderedTestGuid, pendingTest.OrderedTest.PatientOrderGuid);

                    if (orderedTest.TaskStatusCode != Utility.GetTaskStatusCodeFromEnum(taskStatus))
                    {
                        return false;
                    }
                }
            }

            return true;
        }
        #endregion

        private void DeleteTableData()
        {
            UnitTestUtil.RunSQL("DELETE FROM PendingSpecimenTest");
            // Delete any XM tests in the PendingBloodUnitTest table
            UnitTestUtil.RunSQL("DELETE FROM PendingBloodUnitTest"); 
        }

        internal class PendingSpecimenTestArranger : BaseTest
        {
            public string DivisionCode { get; set; }
            public Patient Patient { get; set; }
            public Guid PatientOrderGuid { get; set; }
            public Guid PatientSpecimenGuid { get; set; }
            // For XM
            public BloodUnit BloodUnit { get; set; }

            public PendingSpecimenTestArranger(bool createWorkLoad)
            {
                DivisionCode = this.TestDivisionCode;
                PatientSpecimenGuid = Guid.NewGuid();
                PatientOrderGuid = Guid.NewGuid();
                Patient = DataCreator.CreateNewPatient();

                DataCreator.InsertPatientTreatment(Patient.PatientGuid, Patient.PatientGuid);
                DataCreator.InsertPatientOrder(PatientOrderGuid, Patient.PatientGuid);
                DataCreator.InsertPatientSpecimen(PatientSpecimenGuid, Patient.PatientGuid, false);

                if (createWorkLoad)
                {
                    DataCreator.CreateWorkloadProcess(WorkloadProcessID.AutomatedPatientABO);
                }
                else
                {
                    var deleteWorkLoadSql = "DELETE FROM WorkloadProcess WHERE VbecsProcessId = " + ((int)WorkloadProcessID.AutomatedPatientABO).ToString();
                    UnitTestUtil.RunSQL(deleteWorkLoadSql);
                }
            }

            public void Arrange(PendingTestStatus pendingTestStatus, OrderableTest orderableTest)
            {
                var statusId = ((int)pendingTestStatus).ToString();
                var resultSetGuid = Guid.NewGuid().ToString();
                var updateFunctionId = ((int)UpdateFunction.UC115SpecimenTabView).ToString();
                string sql = "";

                var orderedTestGuid = Guid.NewGuid();
                DataCreator.InsertOrderedTest(orderedTestGuid, PatientOrderGuid, PatientSpecimenGuid, orderableTest, TaskStatus.PendingReview);

                if (orderableTest == OrderableTest.ABORh)
                {
                    sql =
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.AntiA).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'4  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.AntiB).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.PlasmaA1Cells).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.PlasmaBCells).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'3  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.AntiD).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'4  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ControlD).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ABOInterp).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'A  ', N'Immucor Echo', N'Testing patient abo/rh (pos)', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")";
                }
                else if (orderableTest == OrderableTest.ABS)
                {
                    sql =
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ABS_AHG_SC1).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ABS_AHG_SC2).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ABS_AHG_SC3).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.AntibodyScreen).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'N  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")";
                }
                else if (orderableTest == OrderableTest.TAS)
                {
                    sql =
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.AntiA_TAS).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.AntiB_TAS).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ABOInterpTAS).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'O  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.AntiD_TAS).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.DControl).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.RhInterpTAS).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'N  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.A1Cells).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'2  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.BCells).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'2  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ABS_TAS_SC1).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'4  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ABS_TAS_SC2).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'4  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.ABS_TAS_SC3).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'4  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.AntibodyScreenTAS).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'P  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")";
                }
                else if (orderableTest == OrderableTest.AGe)
                {
                    sql =
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', 225, CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'0  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', 230, CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'N  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")";
                }
                else if (orderableTest == OrderableTest.DAT)
                {
                    sql =
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.DAT_Poly_AHG).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'2  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', " + ((int)TestType.DAT_Poly_CC).ToString() + ", CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'3  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")" +
                    "INSERT [dbo].[PendingSpecimenTest] ([SpecimenTestGuid], [OrderedTestGuid], [PatientSpecimenGuid], [BloodTestTypeId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentName], [AutoInstrumentComments], [DivisionCode], [PendingTestStatusId], [RejectionComment], [SpecimenResultSetGuid], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId]) VALUES (NULL, N'" + orderedTestGuid.ToString() + "', N'" + PatientSpecimenGuid.ToString() + "', 55, CAST(N'2016-01-27 12:30:42.000' AS DateTime), N'DNS   DOBRAK', N'P  ', N'Immucor Echo', N'', N'" + DivisionCode + "', " + statusId + ", NULL, N'" + resultSetGuid + "', CAST(N'2016-01-27 21:09:01.183' AS DateTime), N'vbecsuser1', " + updateFunctionId + ")";
                }

                if (!string.IsNullOrEmpty(sql))
                {
                    UnitTestUtil.RunSQL(sql);
                }
            }

            public void ArrangeXM(PendingTestStatus pendingTestStatus, OrderableTest orderableTest, string xmInterpResultCode /*I = Incompatible, C = Compatible*/)
            {
                var statusId = ((int)pendingTestStatus).ToString();
                var resultSetGuid = Guid.NewGuid().ToString();
                var updateFunctionId = ((int)UpdateFunction.UC115SpecimenTabView).ToString();
                string sql = "";

                Guid orderedTestGuid = Guid.NewGuid();
                Guid orderedComponentGuid = Guid.NewGuid();
                DataCreator.InsertOrderedComponent(orderedComponentGuid, PatientOrderGuid, Common.ComponentClass.RBC, 1, PatientSpecimenGuid);
                DataCreator.InsertOrderedTest(orderedTestGuid, PatientOrderGuid, PatientSpecimenGuid, orderableTest);

                var bloodUnitStatusGuid = Guid.NewGuid();
                var bloodProductGuid = DataCreator.GetRandomBloodProduct(false, Common.ComponentClass.ConversionClass, "").BloodProductGuid;
                var bloodUnitGuid = Guid.NewGuid();
                var orderedUnitGuid = Guid.NewGuid();

                BloodUnit = DataCreator.CreateBloodUnit(bloodUnitGuid, bloodUnitStatusGuid, Patient.PatientGuid, bloodProductGuid, false);
                DataCreator.InsertOrderedUnit(orderedUnitGuid, orderedComponentGuid, BloodUnit.BloodUnitGuid);

                sql =
                    "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [OrderedUnitGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) " +
                    "VALUES (NULL, N'" + BloodUnit.BloodUnitGuid + "', " + ((int)TestType.XM_IS).ToString() + ", N'" + BloodUnit.LabelUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'2  ', N'instrument comment', NULL, '" + orderedComponentGuid.ToString() + "', '" + orderedUnitGuid.ToString() + "', N'ProVue', N'" + DivisionCode + "  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', " + updateFunctionId + ", N'" + resultSetGuid + "') " +
                    "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [OrderedUnitGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) " +
                    "VALUES (NULL, N'" + BloodUnit.BloodUnitGuid + "', " + ((int)TestType.XM_AHG).ToString() + ", N'" + BloodUnit.LabelUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'2  ', N'instrument comment', NULL, '" + orderedComponentGuid.ToString() + "', '" + orderedUnitGuid.ToString() + "', N'ProVue', N'" + DivisionCode + "  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', " + updateFunctionId + ", N'" + resultSetGuid + "') " +
                    "INSERT [dbo].[PendingBloodUnitTest] ([BloodUnitTestGuid], [BloodUnitGuid], [BloodTestTypeId], [BloodUnitId], [TestDate], [TestTechId], [TestResultId], [AutoInstrumentComments], [OrderedTestGuid], [OrderedComponentGuid], [OrderedUnitGuid], [AutoInstrumentName], [DivisionCode], [PendingTestStatusId], [LastUpdateDate], [LastUpdateUser], [LastUpdateFunctionId], [UnitResultSetGuid]) " +
                    "VALUES (NULL, N'" + BloodUnit.BloodUnitGuid + "', " + ((int)TestType.XM_Interp).ToString() + ", N'" + BloodUnit.LabelUnitId + "', CAST(N'2015-12-08 12:30:00.000' AS DateTime), N'DNS   VBECS1', N'" + xmInterpResultCode + "  ', N'instrument comment', NULL, '" + orderedComponentGuid.ToString() + "', '" + orderedUnitGuid.ToString() + "', N'ProVue', N'" + DivisionCode + "  ', " + statusId + ", CAST(N'2015-12-08 12:31:00.000' AS DateTime), N'DNS   VBECS1', " + updateFunctionId + ", N'" + resultSetGuid + "') ";


                UnitTestUtil.RunSQL(sql);
            }
        }
    }
}


#endif



